use reqwest::Url;

use crate::full_text_parser::error::FullTextParserError;
use crate::util::Util;
use crate::{FtrConfigEntry, FullTextParser};

pub struct CleanedHtml {
    pub html: String,
    pub thumbnail: Option<String>,
}

/// Re-use crate internals to clean HTML fragments of articles before
/// further processing:
/// - replace H1 with H2
/// - rename all font nodes to span
/// - unwrap noscript images
/// - strip noscript tags
/// - fix lazy-load images
/// - fix iframe size
/// - remove onclick of links
/// - strip elements using Readability.com and Instapaper.com ignore class names
/// - strip elements that contain style="display: none;"
/// - strip styles
/// - strip input elements
/// - strip comments
/// - strip scripts
/// - strip all external css and fonts
/// - complete relative urls
/// - simplify nested elements
///
/// # Arguments
///
/// * `html` - HTML content
/// * `base_url` - URL used to complete relative URLs
///
pub fn clean_html_fragment(
    html_fragment: &str,
    base_url: &Url,
) -> Result<CleanedHtml, FullTextParserError> {
    libxml::tree::node::set_node_rc_guard(10);

    let empty_config = FtrConfigEntry::default();
    let document = FullTextParser::parse_html(html_fragment, None, &empty_config)?;
    let xpath_ctx = FullTextParser::get_xpath_ctx(&document)?;
    let thumbnail = FullTextParser::check_for_thumbnail(&xpath_ctx);
    FullTextParser::prep_content(&xpath_ctx, None, &empty_config, base_url, &document, None);
    if let Some(mut root) = document.get_root_element() {
        Util::clean_headers(&mut root);
        Util::replace_schema_org_orbjects(&mut root);
        Util::clean_conditionally(&mut root, "fieldset");
        Util::clean_conditionally(&mut root, "table");

        FullTextParser::remove_share_elements(&mut root);
        FullTextParser::clean_attributes(&mut root)?;
        FullTextParser::remove_single_cell_tables(&mut root);
        FullTextParser::remove_extra_p_and_div(&mut root);
        FullTextParser::remove_empty_nodes(&mut root);
    }
    FullTextParser::prevent_self_closing_tags(&xpath_ctx)?;
    FullTextParser::post_process_document(&document)?;

    let content_node = if let Some(root) = document.get_root_element() {
        if root.get_name() == "body" {
            Some(root)
        } else if let Some(body) = Util::get_first_element_by_tag_name(&root, "body") {
            Some(body)
        } else {
            Some(root)
        }
    } else {
        None
    }
    .ok_or(FullTextParserError::Xml)?;

    Ok(CleanedHtml {
        html: document.node_to_string(&content_node),
        thumbnail,
    })
}

#[cfg(test)]
mod tests {
    use super::clean_html_fragment;
    use reqwest::Url;

    #[test]
    fn rethinking_window_management() {
        let html = r#"<div xmlns="http://www.w3.org/1999/xhtml"><p>Window management is one of those areas I’m fascinated with because even after 50 years, nobody’s fully cracked it yet. <a href="https://en.wikipedia.org/wiki/Xerox_Alto">Ever since the dawn of time</a> we’ve relied on the <a href="https://en.wikipedia.org/wiki/WIMP_(computing)">window metaphor</a> as the primary way of multitasking on the desktop. In this metaphor, each app can spawn one or more rectangular windows, which are stacked by most recently used, and moved or resized manually.</p>
        <figure class="wp-caption alignnone" id="attachment_9934" style="width: 660px;"><img alt="" class="wp-image-9934 size-large" height="371" src="https://blogs.gnome.org/tbernard/files/2023/07/floating-windows2-1-1024x576.png" width="660"/><figcaption class="wp-caption-text" id="caption-attachment-9934">Overlapping windows can get messy quickly</figcaption></figure>
        <p>The traditional windowing system works well as long as you only have a handful of small windows, but issues emerge as soon the number and size of the windows grows. As new windows are opened, existing ones are obscured, sometimes completely hiding them from view. Or, when you open a maximized window, suddenly every other window is hidden.</p>
        <p>Over the decades, different OSes have added different tools and workflows to deal with these issues, including workspaces, taskbars, and switchers. However, the basic primitives have not changed <a href="http://toastytech.com/guis/alto3.html">since the 70s</a> and, as a result, the issues have never gone away.</p>
        <p>While most of us are used to this system and its quirks, that doesn’t mean it’s without problems. This is especially apparent when you do user research with people who are new to computing, including children and older people. Manually placing and sizing windows can be fiddly work, and requires close attention and precise motor control. It’s also what we jokingly refer to as <a href="https://zachholman.com/posts/shit-work">shit work</a>: it is work that the user has to do, which is generated by the system itself, and has no other purpose.</p>
        <p>Most of the time you don’t care about exact window sizes and positions and just want to see the windows that you need for your current task. Often that’s just a single, maximized window. Sometimes it’s two or three windows next to each other. It’s incredibly rare that you need a dozen different overlapping windows. Yet this is what you end up with by default today, when you simply use the computer, opening apps as you need them. Messy is the default, and it’s up to you to clean it up.</p>
        <h4>What about tiling?</h4>
        <p>Traditional tiling window managers solve the hidden window problem by preventing windows from overlapping. While this works well in some cases, it falls short as a general replacement for stacked, floating windows. The first reason for this is that tiling window managers size windows according to the amount of available screen space, yet most apps are designed to be used at a certain size and aspect ratio. For example, chat apps are inherently narrow and end up having large amounts of empty space at large sizes. Similarly, reading a PDF in a tiny window is not fun.</p>
        <figure class="wp-caption alignnone" id="attachment_9949" style="width: 660px;"><img alt="" class="size-large wp-image-9949" height="371" src="https://blogs.gnome.org/tbernard/files/2023/07/forge2-1024x576.png" width="660"/><figcaption class="wp-caption-text" id="caption-attachment-9949">GNOME 44 with the “<a href="https://extensions.gnome.org/extension/4481/forge">Forge</a>” tiling extension. Just because windows <em>can</em> be tall and narrow doesn’t mean they should be :)</figcaption></figure>
        <p>Another issue with tiling window manager is that they place new windows in seemingly arbitrary positions. This is a consequence of them not having knowledge about the content of a window or the context in which it is being used, and leads to having to manually move or resize windows after the fact, which is exactly the kind of fiddling we want to avoid in the first place.</p>
        <div class="wp-video" style="width: 660px;"><!--[if lt IE 9]><script>document.createElement('video');</script><![endif]-->
        <video class="wp-video-shortcode" controls="" height="495" id="video-9880-1" preload="metadata" width="660"><source src="https://blogs.gnome.org/tbernard/files/2023/07/ipad-tiling-shortened.webm?_=1" type="video/webm"/><a href="https://blogs.gnome.org/tbernard/files/2023/07/ipad-tiling-shortened.webm">https://blogs.gnome.org/tbernard/files/2023/07/ipad-tiling-shortened.webm</a></video></div>
        <p>More constrained tiling window managers such as on iPadOS are interesting in that they’re more purposeful (you always intentionally create the tiling groups). However, this approach only allows tiling two windows side-by-side, and does not scale well to larger screens.</p>
        <h4>History</h4>
        <p>This topic has been of interest to the design team for a <a href="https://cassidyjames.com/blog/gnome-west-coast-summit">very long time</a>. I remember discussing it with Jakub at my first GUADEC in 2017, and there have been countless discussions, ideas, and concepts since. Some particular milestones in our thinking were the concept work leading up to <a href="https://blogs.gnome.org/shell-dev/2020/12/18/gnome-shell-ux-plans-for-gnome-40">GNOME 40</a> in <a href="https://blogs.gnome.org/tbernard/2019/12/02/gnome-shell-hackfest-2019">2019</a> and 2020, and the design sessions at the <a href="https://blogs.gnome.org/tbernard/2022/07/30/berlin-mini-guadec-2022">Berlin Mini GUADEC in 2022</a> and the Brno hackfest in 2023.</p>
        <figure class="wp-caption alignnone" id="attachment_10020" style="width: 660px;"><img alt="" class="wp-image-10020 size-large" height="495" src="https://blogs.gnome.org/tbernard/files/2023/07/2023-04-26-11-24-43-741-1024x768.jpg" width="660"/><figcaption class="wp-caption-text" id="caption-attachment-10020">Tiling BoF in Brno during the HDR hackfest. Left to right: Robert Mader, Marco Trevisan, Georges Stavracase, Jakub Steiner and Allan Day (remote), Florian Müllner, Jonas Dreßler</figcaption></figure>
        <p>I personally have a bit of a tradition working on this problem for at least a few weeks per year. For example, during the first lockdown in 2020 I spent quite a bit of time trying to envision a tiling-first version of GNOME Shell.</p>
        <figure class="wp-caption alignnone" id="attachment_9895" style="width: 660px;"><img alt="" class="size-large wp-image-9895" height="413" src="https://blogs.gnome.org/tbernard/files/2023/07/prometheus-1024x640.png" width="660"/><figcaption class="wp-caption-text" id="caption-attachment-9895">2020 mockup for a tiling-first GNOME Shell. More mockups in the <a href="https://gitlab.gnome.org/Teams/Design/os-mockups/-/tree/master/mobile-shell/tiling">OS mockups repo on Gitlab</a>.</figcaption></figure>
        <h4>Problems with our current tiling</h4>
        <p>GNOME has had basic tiling functionality since early in the GNOME 3 series. While this is nice to have, it has obvious limitations:</p>
        <ul>
        <li>It’s completely manual</li>
        <li>Only 2 windows are supported, and the current implementation is not extensible to more complex layouts</li>
        <li>Tiled windows are not grouped in the window stack, so both windows are not raised simultaneously and other windows get in the way</li>
        <li>Workspaces are manual, and not integrated into the workflow</li>
        </ul>
        <figure class="wp-caption alignnone" id="attachment_9957" style="width: 660px;"><img alt="" class="wp-image-9957 size-large" height="371" src="https://blogs.gnome.org/tbernard/files/2023/07/tiling-status-quo-1024x576.png" width="660"/><figcaption class="wp-caption-text" id="caption-attachment-9957">Because tiled windows are currently mixed with overlapping floating windows they’re not really helping make things less messy in practice.</figcaption></figure>
        <p>We’ve wanted <a href="https://gitlab.gnome.org/Teams/Design/os-mockups/-/issues/58">more powerful tiling</a> <a href="https://gitlab.gnome.org/Teams/Design/os-mockups/-/issues/169">for years</a>, but there has not been much progress due to the huge amount of work involved on the technical side and the lack of a clear design direction we were happy with. We now finally feel like the design is at a stage where we can take concrete next steps towards making it happen, which is very exciting!</p>
        <h4>Get out of my way</h4>
        <p>The key point we keep coming back to with this work is that, if we do add a new kind of window management to GNOME, it needs to be good enough to be the default. We don’t want to add yet another manual opt-in tool that doesn’t solve the problems the majority of people face.</p>
        <p>To do this we landed on a number of high level ideas:</p>
        <ul>
        <li>Automatically do what people probably want, allow adjusting if needed</li>
        <li>Make use of workspaces as a fully integrated part of the workflow</li>
        <li>Richer metadata from apps to allow for better integration</li>
        </ul>
        <p>Our current concept imagines windows having three potential layout states:</p>
        <ul>
        <li><strong>Mosaic</strong>, a new window management mode which combines the best parts of tiling and floating</li>
        <li><strong>Edge Tiling</strong>, i.e. windows splitting the screen edge-to-edge</li>
        <li><strong>Floating</strong>, the classic stacked windows model</li>
        </ul>
        <div class="wp-video" style="width: 660px;"><video class="wp-video-shortcode" controls="" height="495" id="video-9880-2" preload="metadata" width="660"><source src="https://blogs.gnome.org/tbernard/files/2023/07/mosaic-open-close.webm?_=2" type="video/webm"/><a href="https://blogs.gnome.org/tbernard/files/2023/07/mosaic-open-close.webm">https://blogs.gnome.org/tbernard/files/2023/07/mosaic-open-close.webm</a></video></div>
        <p>Mosaic is the default behavior. You open a window, it opens centered on the screen at a size that makes the most sense for the app. For a web browser that might be maximized, for a weather app maybe only 700×500 pixels.</p>
        <div class="wp-video" style="width: 660px;"><video class="wp-video-shortcode" controls="" height="495" id="video-9880-3" preload="metadata" width="660"><source src="https://blogs.gnome.org/tbernard/files/2023/07/mosaic-maximize.webm?_=3" type="video/webm"/><a href="https://blogs.gnome.org/tbernard/files/2023/07/mosaic-maximize.webm">https://blogs.gnome.org/tbernard/files/2023/07/mosaic-maximize.webm</a></video></div>
        <p>As you open more windows, the existing windows move aside to make room for the new ones. If a new window doesn’t fit (e.g. because it wants to be maximized) it moves to its own workspace. If the window layout comes close to filling the screen, the windows are automatically tiled.</p>
        <div class="wp-video" style="width: 660px;"><video class="wp-video-shortcode" controls="" height="495" id="video-9880-4" preload="metadata" width="660"><source src="https://blogs.gnome.org/tbernard/files/2023/07/mosaic-tile2.webm?_=4" type="video/webm"/><a href="https://blogs.gnome.org/tbernard/files/2023/07/mosaic-tile2.webm">https://blogs.gnome.org/tbernard/files/2023/07/mosaic-tile2.webm</a></video></div>
        <p>You can also manually tile windows. If there’s enough space, other windows are left in a mosaic layout. However, if there’s not enough space for this mosaic layout, you’re prompted to pick another window to tile alongside.</p>
        <div class="wp-video" style="width: 660px;"><video class="wp-video-shortcode" controls="" height="495" id="video-9880-5" preload="metadata" width="660"><source src="https://blogs.gnome.org/tbernard/files/2023/07/mosaic-tile3.webm?_=5" type="video/webm"/><a href="https://blogs.gnome.org/tbernard/files/2023/07/mosaic-tile3.webm">https://blogs.gnome.org/tbernard/files/2023/07/mosaic-tile3.webm</a></video></div>
        <p>You’re not limited to tiling just two windows side by side. Any tile (or the remaining space) can be split by dragging another window over it, and freely resized as the window minimum sizes allow.</p>
        <div class="wp-video" style="width: 660px;"><video class="wp-video-shortcode" controls="" height="495" id="video-9880-6" preload="metadata" width="660"><source src="https://blogs.gnome.org/tbernard/files/2023/07/mosaic-vertical-tile.webm?_=6" type="video/webm"/><a href="https://blogs.gnome.org/tbernard/files/2023/07/mosaic-vertical-tile.webm">https://blogs.gnome.org/tbernard/files/2023/07/mosaic-vertical-tile.webm</a></video></div>
        <p>There are always going to be cases that require placing a window in a specific position on the screen. The new system allows windows to be used with the classic floating behavior, on a layer above the mosaic/tiling windows. However, we think that this floating behaviour is going to be a relatively uncommon, similar to the existing “always on top” behavior that we have today.</p>
        <p>There’s of course much more to this, but hopefully this gives an idea of what we have in mind in terms of behavior.</p>
        <h4>New window metadata</h4>
        <p>As mentioned above, to avoid the pitfalls of traditional tiling window managers we need more information from windows about their content. Windows can already set a fixed size and they have an implicit minimum size, but to build a great tiling experience we need more.</p>
        <figure class="wp-caption alignnone" id="attachment_9972" style="width: 660px;"><img alt="" class="wp-image-9972 size-large" height="371" src="https://blogs.gnome.org/tbernard/files/2023/07/huge-clocks-1024x576.png" width="660"/><figcaption class="wp-caption-text" id="caption-attachment-9972">Some apps should probably never be maximized/tiled on a 4K monitor…</figcaption></figure>
        <p>One important missing piece is having information on the maximum desired size of a window. This is the size beyond which the window content stops looking good. Not having this information is one of the reasons that traditional tiling window managers have issues, especially on larger screens. This maximum size would not be a hard limit and manual resizing would still be possible. Instead, the system would use the maximum size as one factor when it calculates an optimal window layout. For example, when tiling to the side of the screen, a window would only grow as wide as its maximum width rather than filling exactly half of the screen.</p>
        <p>In addition, it’d be helpful to know the range of ideal sizes where an app works best. While an app may technically work at mobile sizes that’s probably not the best way to use that app if you have a large display. To stay with our chat example, you probably want to avoid folding the sidebar if it can be avoided, so the range of ideal sizes would be between the point where it becomes single pane and its maximum usable size.</p>
        <p>Ideally these properties could be set dynamically depending on the window content. For example, a spreadsheet with a lot of columns but few rows could have a wider ideal size than one with lots of rows.</p>
        <p>Depending on apps using new system APIs can be challenging and slow — it’s not easy to move the entire ecosystem! However, we think there’s a good chance of success in this case, due to the simplicity and universal usefulness of the API.</p>
        <h4>Next steps</h4>
        <p>At the Brno hackfest in April we had an initial discussion with GNOME Shell developers about many of the technical details. There is tentative agreement that we want to move in the direction outlined in this post, but there’s still a lot of work ahead.</p>
        <p>On the design side, the biggest uncertainty is the mosaic behavior — it’s a novel approach to window management without much prior art. That’s exciting, but also makes it a bit risky to jump head-first into implementation. We’d like to do user research to validate some of our assumptions on different aspects of this, but it’s the kind of project that’s very difficult to test outside of an actual prototype that’s usable day to day.</p>
        <p>If you’d like to get involved with this initiative, one great way to help out would be to work on an extension that implements (parts of) the mosaic behavior for testing and refining the interactions. If you’re interested in this, please reach out :)</p>
        <p>There’s no timeline or roadmap at this stage, but it’s definitely 46+ material and likely to take multiple cycles. There are individual parts of this that could be worked on independently ahead of the more contingent pieces, for example tiling groups or new window metadata. Help in any of these areas would be appreciated.</p>
        <p><em>This post is summarizing collaborative work over the past years by the entire design team (Allan Day, Jakub Steiner, Sam Hewitt, et al). In particular, thanks to Jakub for the awesome animations bringing the behaviors to life!</em></p></div>"#;

        let url =
            Url::parse("https://blogs.gnome.org/tbernard/2023/07/26/rethinking-window-management/")
                .unwrap();
        _ = clean_html_fragment(html, &url).unwrap();
    }

    #[test]
    fn finshots() {
        let html = r#"<img src="https://cdn.finshots.app/images/2023/03/Design-8-Amul.jpg" alt="Amul, Cola and Atta???"><p><em>In today’s Finshots, we discuss Amul’s pathway to becoming more than just a dairy brand.</em></p><!--kg-card-begin: hr--><hr><!--kg-card-end: hr--><h3 id="the-story">The Story</h3><p>The ₹61,000 crore Amul has a new leader — Jayen Mehta. And <a href="https://economictimes.indiatimes.com/industry/cons-products/fmcg/amul-readies-to-take-on-the-cokes-of-the-world-says-md-jayen-mehta/articleshow/98843180.cms" rel="noopener">he says</a> he wants to transform the dairy giant into a veritable FMCG behemoth. Think atta to compete with ITC’s Aashirvaad. Biscuits that creep into Britannia’s territory and even carbonated beverages to take on the might of Coca-Cola and Pepsi.</p><p>Now, you might have seen some of these products on your supermarket shelves already. Because they’re not exactly brand new launches. Amul has slowly been testing the waters over the past few years. And now, it just wants to double down on this diversification.</p><p>But before we get into why and how let’s rewind a bit to understand Amul’s history.</p><p>The <a href="https://www.amuldairy.com/history.php#:~:text=It%20was%20formally%20registered%20on,about%20250%20liters%20a%20day." rel="noopener">story</a> begins in 1945. The milk farmers at Anand in Gujarat’s Kaira (now Kheda) district were miserable. The entire market was controlled by one entity — Polson’s Dairy. See, the government had launched the Bombay Milk Scheme where milk had to be sent from Anand to Bombay. And since milk is perishable, it couldn’t be quickly transported across the country without getting spoilt. So the milk had to be pasteurised at Anand itself. And considering Polson had the factories, it emerged as the winner and it began to dictate prices to the farmers. They paid peanuts and Polson’s and the middlemen pocketed all the profits from the sales.</p><p>But then came Sardar Vallabhai Patel, the Iron Man of India, who rallied the farmers into setting up a cooperative. He wanted them to work together and pool their resources. A bigger unit meant that they could dictate their own terms. The farmers went on strike. Bombay ran out of milk. And finally, the Kaira District Co-operative Milk Producers’ Union or Amul was born. They kicked Polsons out of the game and started pasteurising milk for the Bombay Milk Scheme in 1948. Two villages, 250 litres of milk. That’s it.</p><p>But soon, there was another problem ― excess milk. See, because of a shortage of cow milk, the Union processed buffalo milk as well. But there came a point where Bombay wasn’t able to absorb this excess milk.</p><p>Enter Dr. Verghese Kurien, a government servant who was deputed to Anand’s experimental creamery. The man chalked out a billion-litre idea of reprocessing excess buffalo milk. And that’s when they decided to set up a factory to churn the raw milk into milk powder and butter. Products that had a longer shelf-life. In 1954, the first step towards the diversification of Amul’s products began.</p><p>Amul became a pan-India movement. And what started as a tiny union of a handful of farmers producing 250 litres of milk a day is now a 3.6 million-strong organisation producing an average of over <a href="https://amul.com/m/organisation" rel="noopener">26 million litres of milk</a> daily.</p><p>So yeah, you can see why consumers like you and me consider Amul synonymous with dairy. There’s a long history and there’s nothing else quite like it.</p><p>Now diversification is a natural strategy for any company, right? No one wants to be dependent on just one product. Also, milk is just a commodity. You can’t really earn too much margin on it. So Amul began to create milk-adjacent products that would add more value to the consumer. These products could be priced higher and make the cooperative more money — cheese, paneer, buttermilk, flavoured shakes, and ice creams were a perfect fit for a dairy company. And the strategy worked. <a href="https://www.financialexpress.com/brandwagon/amul-adds-zing-to-its-kitty-broadens-product-portfolio/2144538/" rel="noopener">In FY19–20</a>, these value-added products actually contributed to 45% of its revenues.</p><p>Now if you think about it, Amul has all the ingredients to succeed with its diversification into non-dairy items like colas, atta, biscuits, and french fries too. It just needs to follow the same playbook, right?</p><p>It has a brand image that has been carefully cultivated over the years. In part due to the iconic Amul girl in the red polka-dotted dress. While other leading brands apportion 8–15% of their total spending on ads, Amul spends <a href="https://www.financialexpress.com/industry/spending-less-on-marketing-is-amuls-winning-formula-heres-why-it-spends-only-1-on-ads/1571085/" rel="noopener">less than 1%</a> on advertisements. And this brand image can come in handy for penetrating the rural markets which typically make up <a href="https://www.livemint.com/industry/retail/rural-paced-ahead-of-urban-in-demand-for-branded-products-kantar-11627555961420.html" rel="noopener">nearly 40%</a> of an FMCG company’s sales. People trust Amul.</p><p>And most importantly, Amul has a massive <a href="https://www.livemint.com/companies/news/how-amul-swung-the-great-indian-milk-run-11594651047495.html" rel="noopener">distribution network</a> it can tap — 10,000 distributors and over a million retailers. Its frozen products like french fries and aloo tikki can simply leverage its existing ice cream cold chain network. Amul really doesn’t need to build new distribution facilities from scratch.</p><p>But here’s the thing. Despite its decades of success selling dairy products, Amul hasn’t quite been able to crack the diversification code. It hasn’t been able to emerge as a true FMCG player yet.</p><p>Take chocolates for instance. Amul actually forayed into the industry way back in the 1970s itself. In fact, it tried the same playbook of setting up a cooperative society for cocoa farming. It wanted to fight Cadbury’s monopoly. It thought it could easily use its existing cold chain network for distribution. It even advertised heavily when colour televisions became popular in India in the 1980s. But nothing worked. Today, Amul has a measly <a href="https://thewire.in/business/the-unfinished-dream-behind-amuls-foray-into-the-chocolate-industry" rel="noopener">3% market share</a> in India.</p><p><a href="https://economictimes.indiatimes.com/amul-plans-to-energise-local-sports-drinks-sector-with-stamina/articleshow/1394953.cms?from=mdr" rel="noopener">In 2006</a>, it launched a sports drink called Stamina. It didn’t see any takers. It shut shop, re-launched the drink a decade later and failed again. Amul even launched a <a href="https://economictimes.indiatimes.com/amul-pizza-loses-curves-tries-angular-design/articleshow/1585437.cms?from=mdr" rel="noopener">frozen pizza</a> in the 2000s! And if you’re surprised at that bit of news, well, that’s because it failed too.</p><p>In 2019, it forayed into butter cookies. And it even took on rivals like Britannia’s Good Day. It <a href="https://www.bqprime.com/business/butter-cookies-frozen-potatoes-and-soon-amul-fruits-and-vegetables" rel="noopener">thought</a>, “Hey, we’re supplying all the butter to these FMCG companies. But they’re actually mixing a lot of palm oil into it. Why not make one of our own?”</p><p>Amul even went on the offensive and launched ad campaigns <a href="https://www.thehindubusinessline.com/companies/amuls-ad-campaign-puts-spotlight-on-real-butter-content-in-cookies/article28691311.ece" rel="noopener">saying </a>that it had ‘25% Amul butter.’ And that everyone else had less than 3%. It said that rivals simply used a flavouring. But despite that ad blitz, Amul hasn’t set the butter cookie segment on fire.</p><p>And in 2020, it launched the Amul Tru seltzer — a carbonated fizzy drink to take on the colas of India. But even this product hasn’t moved the needle.</p><p>Basically, almost everything other than the value-added dairy products hasn’t quite worked out for Amul. Its brand or distribution hasn’t helped it. So will it be different this time under new leadership? We don’t know.</p><p>Or maybe Amul should just do what it does best and focus on getting more of the dairy pie? After all, <a href="https://www.bqprime.com/business/butter-cookies-frozen-potatoes-and-soon-amul-fruits-and-vegetables" rel="noopener">only 30%</a> of the $110-billion dairy sector is organized even today.</p><p>Can Amul crack the code for non-dairy FMCG products? What do you think?</p><p>Until then…</p><p><em>Don't forget to share this article on <a href="https://api.whatsapp.com/send?text=An%20explainer%20on%20Amul's%20diversification%20bid%20https://bit.ly/3TPwkGc">WhatsApp</a>, <a href="https://www.linkedin.com/shareArticle?mini=true&amp;url=https://finshots.in/archive/amul-cola-and-atta">LinkedIn</a> and <a href="https://twitter.com/intent/tweet?url=https://bit.ly/3FW2NVF&amp;via=finshots&amp;text=An%20explainer%20on%20Amul's%20diversification%20bid">Twitter</a></em></p><!--kg-card-begin: hr--><hr><!--kg-card-end: hr--><h3 id="ditto-insights-why-millennials-should-buy-a-term-plan">Ditto Insights: Why Millennials should buy a term plan</h3><p>According to a survey, only 17% of Indian millennials (25–35 yrs) have bought term insurance. The actual numbers are likely even lower.</p><p>And the more worrying fact is that 55% hadn’t even heard of term insurance!</p><p>So why is this happening?</p><p>One common misconception is the dependent conundrum. Most millennials we spoke to want to buy a term policy because they want to cover their spouse and kids. And this makes perfect sense. After all, in your absence you want your term policy to pay out a large sum of money to cover your family’s needs for the future. But these very same people don’t think of their parents as dependents even though they support them extensively. I remember the moment it hit me. I routinely send money back home, but I had never considered my parents as my dependents. And when a colleague spoke about his experience, I immediately put two and two together. They were dependent on my income and my absence would most certainly affect them financially. So a term plan was a no-brainer for me.</p><p>There’s another reason why millennials should probably consider looking at a term plan — Debt. Most people we spoke to have home loans, education loans and other personal loans with a considerable interest burden. In their absence, this burden would shift to their dependents. It’s not something most people think of, but it happens all the time.</p><p>Finally, you actually get a pretty good bargain on term insurance prices when you’re younger. The idea is to pay a nominal sum every year (something that won’t burn your pocket) to protect your dependents in the event of your untimely demise. And this fee is lowest when you’re young.</p><p>So if you’re a millennial and you’re reading this, maybe you should reconsider buying a term plan. And don’t forget to talk to us at Ditto while you’re at it. We only have a limited number of slots everyday, so make sure you book your appointment at the earliest:</p><p>1. Just head to our website by clicking on the <a href="https://joinditto.in/?utm_source=Finshots&amp;utm_medium=Newsletter&amp;utm_campaign=30-11-2022&amp;utm_term=CA_T&amp;utm_content=Insights" rel="noopener"><em>link here</em></a></p><p>2. Click on “Book a FREE call”</p><p>3. Select Term Insurance</p><p>4. Choose the date &amp; time as per your convenience and RELAX!</p>"#;
        let url = Url::parse("https://finshots.in").unwrap();
        let res = clean_html_fragment(html, &url).unwrap();

        assert!(res.html.len().abs_diff(12_000) < 200);
        assert_eq!(
            res.thumbnail.as_deref(),
            Some("https://cdn.finshots.app/images/2023/03/Design-8-Amul.jpg")
        )
    }

    #[test]
    fn pointieststick() {
        let html = r#"
        <p>I recently went on Brodie Robertson&#8217;s Tech Over Tea channel for a second time. I guess I didn&#8217;t succeed at pissing him off enough on the first go-around, because he invited me back! Let&#8217;s see if I did a better job of it this time by telling him he was using Arch wrong. <img src="https://s0.wp.com/wp-content/mu-plugins/wpcom-smileys/twemoji/2/72x72/1f600.png" alt="😀" class="wp-smiley" style="height: 1em; max-height: 1em;" /></p>
        <p>Anyway, Brodie was a fantastic host, and we talked about a number of topics such as KDE&#8217;s position in the world, institutional continuity, fundraising and financial stability, the difficulty of reporting and triaging bug, the challenges of packaging software, and windows that block WiFi signals.</p>
        <p>I hope you enjoy it!</p>
        <figure class="wp-block-embed is-type-video is-provider-youtube wp-block-embed-youtube wp-embed-aspect-16-9 wp-has-aspect-ratio"><div class="wp-block-embed__wrapper">
        <iframe class="youtube-player" width="1100" height="619" src="https://www.youtube.com/embed/qJZ2V5FmgO8?version=3&#038;rel=1&#038;showsearch=0&#038;showinfo=1&#038;iv_load_policy=1&#038;fs=1&#038;hl=en&#038;autohide=2&#038;wmode=transparent" allowfullscreen="true" style="border:0;" sandbox="allow-scripts allow-same-origin allow-popups allow-presentation allow-popups-to-escape-sandbox"></iframe>
        </div></figure>
        <p>And here&#8217;s the link I mention at the end: <a href="https://kde.org/community/donations">https://kde.org/community/donations</a> <img src="https://s0.wp.com/wp-content/mu-plugins/wpcom-smileys/twemoji/2/72x72/1f642.png" alt="🙂" class="wp-smiley" style="height: 1em; max-height: 1em;" /> </p>
        "#;

        let url = Url::parse("https://pointieststick.com").unwrap();
        let res = clean_html_fragment(html, &url).unwrap();

        assert_eq!(res.thumbnail, None);
        assert!(res.html.contains("iframe"));
    }

    #[test]
    fn cartographerstale() {
        let html = r#"
        <p class="button-wrapper" data-attrs="{&quot;url&quot;:&quot;https://www.cartographerstale.com/subscribe?&quot;,&quot;text&quot;:&quot;Subscribe now&quot;,&quot;action&quot;:null,&quot;class&quot;:null}" data-component-name="ButtonCreateButton"><a class="button primary" href="https://www.cartographerstale.com/subscribe?"><span>Subscribe now</span></a></p><p>To no one's surprise, I love maps. And like any map enthusiast, I've spent a lot of time reading about map projections. Understanding them is key to being able to interpret a map properly.</p><p>The earth, with its quasi-spherical shape, faces a mathematical problem that has no solution. The sphere, as a geometric figure, cannot be developed on a plane without incurring some kind of distortion. It is not that we have not found the right way to do it, it is that it is not possible. We will never be able to.</p><p>That does not detract from the fact that there can be good approximations, although all of them have some compromise. An easy way to understand it is that there are three fundamental characteristics that cannot be fulfilled simultaneously, so any projection has to reach a compromise between the three:</p><ul><li><p><strong>Distance</strong>: A world map that can preserve the distance between any two points once translated to the plane is considered an <strong>equidistant projection</strong>.</p></li><li><p><strong>The areas</strong>: When the representation on the plane can maintain the areas of a polygon, for example each country, then we speak of an <strong>equivalent projection</strong>.</p></li><li><p><strong>Angles</strong>: This characteristic refers to the angles and, therefore, the shapes of each polygon that we find on the map, for example the outline of a country. If the projection manages to maintain this peculiarity, then we call it <strong>conformal projection</strong>.</p></li></ul><p>Geometry tells us that, since the sphere is not developable in the plane, it is impossible to find a cartographic projection that is simultaneously equidistant, equivalent, and conformal. So we can state categorically that there is no such thing as a <em>correct</em> projection. The most we can aspire to is to have multiple projections available and to use the most relevant ones for each occasion.</p><p>Here are some examples of popular projections, and some information to understand them a little better</p><h3><strong>Mercator, a projection with a utility</strong></h3><div class="captioned-image-container"><figure><a class="image-link image2 is-viewable-img" target="_blank" href="https://substackcdn.com/image/fetch/$s_!zMTj!,f_auto,q_auto:good,fl_progressive:steep/https%3A%2F%2Fsubstack-post-media.s3.amazonaws.com%2Fpublic%2Fimages%2F7284cd3b-ec32-498e-8a4f-32212ecd4880_3500x2223.jpeg" data-component-name="Image2ToDOM"><div class="image2-inset"><picture><source type="image/webp" srcset="https://substackcdn.com/image/fetch/$s_!zMTj!,w_424,c_limit,f_webp,q_auto:good,fl_progressive:steep/https%3A%2F%2Fsubstack-post-media.s3.amazonaws.com%2Fpublic%2Fimages%2F7284cd3b-ec32-498e-8a4f-32212ecd4880_3500x2223.jpeg 424w, https://substackcdn.com/image/fetch/$s_!zMTj!,w_848,c_limit,f_webp,q_auto:good,fl_progressive:steep/https%3A%2F%2Fsubstack-post-media.s3.amazonaws.com%2Fpublic%2Fimages%2F7284cd3b-ec32-498e-8a4f-32212ecd4880_3500x2223.jpeg 848w, https://substackcdn.com/image/fetch/$s_!zMTj!,w_1272,c_limit,f_webp,q_auto:good,fl_progressive:steep/https%3A%2F%2Fsubstack-post-media.s3.amazonaws.com%2Fpublic%2Fimages%2F7284cd3b-ec32-498e-8a4f-32212ecd4880_3500x2223.jpeg 1272w, https://substackcdn.com/image/fetch/$s_!zMTj!,w_1456,c_limit,f_webp,q_auto:good,fl_progressive:steep/https%3A%2F%2Fsubstack-post-media.s3.amazonaws.com%2Fpublic%2Fimages%2F7284cd3b-ec32-498e-8a4f-32212ecd4880_3500x2223.jpeg 1456w" sizes="100vw"><img src="https://substackcdn.com/image/fetch/$s_!zMTj!,w_1456,c_limit,f_auto,q_auto:good,fl_progressive:steep/https%3A%2F%2Fsubstack-post-media.s3.amazonaws.com%2Fpublic%2Fimages%2F7284cd3b-ec32-498e-8a4f-32212ecd4880_3500x2223.jpeg" width="1456" height="925" data-attrs="{&quot;src&quot;:&quot;https://substack-post-media.s3.amazonaws.com/public/images/7284cd3b-ec32-498e-8a4f-32212ecd4880_3500x2223.jpeg&quot;,&quot;srcNoWatermark&quot;:null,&quot;fullscreen&quot;:null,&quot;imageSize&quot;:null,&quot;height&quot;:925,&quot;width&quot;:1456,&quot;resizeWidth&quot;:null,&quot;bytes&quot;:null,&quot;alt&quot;:null,&quot;title&quot;:null,&quot;type&quot;:null,&quot;href&quot;:null,&quot;belowTheFold&quot;:false,&quot;topImage&quot;:true,&quot;internalRedirect&quot;:null,&quot;isProcessing&quot;:false,&quot;align&quot;:null,&quot;offset&quot;:false}" class="sizing-normal" alt="" srcset="https://substackcdn.com/image/fetch/$s_!zMTj!,w_424,c_limit,f_auto,q_auto:good,fl_progressive:steep/https%3A%2F%2Fsubstack-post-media.s3.amazonaws.com%2Fpublic%2Fimages%2F7284cd3b-ec32-498e-8a4f-32212ecd4880_3500x2223.jpeg 424w, https://substackcdn.com/image/fetch/$s_!zMTj!,w_848,c_limit,f_auto,q_auto:good,fl_progressive:steep/https%3A%2F%2Fsubstack-post-media.s3.amazonaws.com%2Fpublic%2Fimages%2F7284cd3b-ec32-498e-8a4f-32212ecd4880_3500x2223.jpeg 848w, https://substackcdn.com/image/fetch/$s_!zMTj!,w_1272,c_limit,f_auto,q_auto:good,fl_progressive:steep/https%3A%2F%2Fsubstack-post-media.s3.amazonaws.com%2Fpublic%2Fimages%2F7284cd3b-ec32-498e-8a4f-32212ecd4880_3500x2223.jpeg 1272w, https://substackcdn.com/image/fetch/$s_!zMTj!,w_1456,c_limit,f_auto,q_auto:good,fl_progressive:steep/https%3A%2F%2Fsubstack-post-media.s3.amazonaws.com%2Fpublic%2Fimages%2F7284cd3b-ec32-498e-8a4f-32212ecd4880_3500x2223.jpeg 1456w" sizes="100vw" fetchpriority="high"></picture><div class="image-link-expand"><div class="pencraft pc-display-flex pc-gap-8 pc-reset"><div class="pencraft pc-reset icon-container restack-image"><svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="lucide lucide-refresh-cw"><path d="M3 12a9 9 0 0 1 9-9 9.75 9.75 0 0 1 6.74 2.74L21 8"></path><path d="M21 3v5h-5"></path><path d="M21 12a9 9 0 0 1-9 9 9.75 9.75 0 0 1-6.74-2.74L3 16"></path><path d="M8 16H3v5"></path></svg></div><div class="pencraft pc-reset icon-container view-image"><svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="lucide lucide-maximize2"><polyline points="15 3 21 3 21 9"></polyline><polyline points="9 21 3 21 3 15"></polyline><line x1="21" x2="14" y1="3" y2="10"></line><line x1="3" x2="10" y1="21" y2="14"></line></svg></div></div></div></div></a><figcaption class="image-caption">(<a href="https://mapasmilhaud.com/mapas-antiguos/mapamundi-de-mercator-copia-de-basilea-1569/">source</a>)</figcaption></figure></div><p>This map is a copy of the world map published by Gerardus Mercator in 1569 and nowadays kept in Basel. Mercator was a cartographer who went down in history for introducing what is possibly the most used and popular projection for centuries, as well as the most hated and criticised in recent decades.</p><p>It is therefore important to understand this projection in its historical context. In the 16th century, Europe was in the midst of the Age of Discovery. Navigation was a key element for the different countries to continue exploring the world, so it had to be efficient and reliable. And this is precisely where this projection really adds great value.</p><p>The Mercator projection is conformal, so angles and shapes are maintained throughout the plane. An indirect quality of this is that all the straight lines that can be drawn on the plane are possible navigational lines that a ship can maintain. This guarantees that it will get from one point on the map to another.</p><p>In other words, when James Cook was on his way to the Pacific Ocean with one of these maps in hand, he was sure that by keeping his course along a straight line on the map he could get easily from Madagascar to India. It is important to stress that this does not guarantee that this line will be the shortest distance, but it will be the one that will require the least additional navigational calculations<a class="footnote-anchor" data-component-name="FootnoteAnchorToDOM" id="footnote-anchor-1" href="\#footnote-1" target="_self">1</a>.</p><h3><strong>Peters, activism with many greys</strong></h3><div class="captioned-image-container"><figure><a class="image-link image2 is-viewable-img" target="_blank" href="https://substackcdn.com/image/fetch/$s_!7huW!,f_auto,q_auto:good,fl_progressive:steep/https%3A%2F%2Fsubstack-post-media.s3.amazonaws.com%2Fpublic%2Fimages%2Fe665974b-0322-4030-83e3-382becc28d61_2422x1588.jpeg" data-component-name="Image2ToDOM"><div class="image2-inset"><picture><source type="image/webp" srcset="https://substackcdn.com/image/fetch/$s_!7huW!,w_424,c_limit,f_webp,q_auto:good,fl_progressive:steep/https%3A%2F%2Fsubstack-post-media.s3.amazonaws.com%2Fpublic%2Fimages%2Fe665974b-0322-4030-83e3-382becc28d61_2422x1588.jpeg 424w, https://substackcdn.com/image/fetch/$s_!7huW!,w_848,c_limit,f_webp,q_auto:good,fl_progressive:steep/https%3A%2F%2Fsubstack-post-media.s3.amazonaws.com%2Fpublic%2Fimages%2Fe665974b-0322-4030-83e3-382becc28d61_2422x1588.jpeg 848w, https://substackcdn.com/image/fetch/$s_!7huW!,w_1272,c_limit,f_webp,q_auto:good,fl_progressive:steep/https%3A%2F%2Fsubstack-post-media.s3.amazonaws.com%2Fpublic%2Fimages%2Fe665974b-0322-4030-83e3-382becc28d61_2422x1588.jpeg 1272w, https://substackcdn.com/image/fetch/$s_!7huW!,w_1456,c_limit,f_webp,q_auto:good,fl_progressive:steep/https%3A%2F%2Fsubstack-post-media.s3.amazonaws.com%2Fpublic%2Fimages%2Fe665974b-0322-4030-83e3-382becc28d61_2422x1588.jpeg 1456w" sizes="100vw"><img src="https://substackcdn.com/image/fetch/$s_!7huW!,w_1456,c_limit,f_auto,q_auto:good,fl_progressive:steep/https%3A%2F%2Fsubstack-post-media.s3.amazonaws.com%2Fpublic%2Fimages%2Fe665974b-0322-4030-83e3-382becc28d61_2422x1588.jpeg" width="1456" height="955" data-attrs="{&quot;src&quot;:&quot;https://substack-post-media.s3.amazonaws.com/public/images/e665974b-0322-4030-83e3-382becc28d61_2422x1588.jpeg&quot;,&quot;srcNoWatermark&quot;:null,&quot;fullscreen&quot;:null,&quot;imageSize&quot;:null,&quot;height&quot;:955,&quot;width&quot;:1456,&quot;resizeWidth&quot;:null,&quot;bytes&quot;:null,&quot;alt&quot;:null,&quot;title&quot;:null,&quot;type&quot;:null,&quot;href&quot;:null,&quot;belowTheFold&quot;:true,&quot;topImage&quot;:false,&quot;internalRedirect&quot;:null,&quot;isProcessing&quot;:false,&quot;align&quot;:null,&quot;offset&quot;:false}" class="sizing-normal" alt="" srcset="https://substackcdn.com/image/fetch/$s_!7huW!,w_424,c_limit,f_auto,q_auto:good,fl_progressive:steep/https%3A%2F%2Fsubstack-post-media.s3.amazonaws.com%2Fpublic%2Fimages%2Fe665974b-0322-4030-83e3-382becc28d61_2422x1588.jpeg 424w, https://substackcdn.com/image/fetch/$s_!7huW!,w_848,c_limit,f_auto,q_auto:good,fl_progressive:steep/https%3A%2F%2Fsubstack-post-media.s3.amazonaws.com%2Fpublic%2Fimages%2Fe665974b-0322-4030-83e3-382becc28d61_2422x1588.jpeg 848w, https://substackcdn.com/image/fetch/$s_!7huW!,w_1272,c_limit,f_auto,q_auto:good,fl_progressive:steep/https%3A%2F%2Fsubstack-post-media.s3.amazonaws.com%2Fpublic%2Fimages%2Fe665974b-0322-4030-83e3-382becc28d61_2422x1588.jpeg 1272w, https://substackcdn.com/image/fetch/$s_!7huW!,w_1456,c_limit,f_auto,q_auto:good,fl_progressive:steep/https%3A%2F%2Fsubstack-post-media.s3.amazonaws.com%2Fpublic%2Fimages%2Fe665974b-0322-4030-83e3-382becc28d61_2422x1588.jpeg 1456w" sizes="100vw" loading="lazy"></picture><div class="image-link-expand"><div class="pencraft pc-display-flex pc-gap-8 pc-reset"><div class="pencraft pc-reset icon-container restack-image"><svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="lucide lucide-refresh-cw"><path d="M3 12a9 9 0 0 1 9-9 9.75 9.75 0 0 1 6.74 2.74L21 8"></path><path d="M21 3v5h-5"></path><path d="M21 12a9 9 0 0 1-9 9 9.75 9.75 0 0 1-6.74-2.74L3 16"></path><path d="M8 16H3v5"></path></svg></div><div class="pencraft pc-reset icon-container view-image"><svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="lucide lucide-maximize2"><polyline points="15 3 21 3 21 9"></polyline><polyline points="9 21 3 21 3 15"></polyline><line x1="21" x2="14" y1="3" y2="10"></line><line x1="3" x2="10" y1="21" y2="14"></line></svg></div></div></div></div></a><figcaption class="image-caption">(<a href="https://mapasmilhaud.com/mapas-politicos/mapa-del-mundo-proyeccion-de-peters-2021/">source</a>)</figcaption></figure></div><p>In the 20th century, the voices against the Mercator projection became louder. The fundamental criticism is that the map designed by Gerardus Mercator had an imperialist intention and that it intentionally under-represented the countries of the Southern Hemisphere, over-representing Europe and North America.</p><p>I always like to emphasize that this is a consequence, not the actual purpose of the map. The Mercator projection is compliant, with the great utility we have just seen, but it is not equivalent, so there are large discrepancies in the area. Contrary to what is often stated, the deformation mainly affects the areas near the poles, which appear with an extreme size, and the areas near the equator, which appear with a minimum size. Of course, South America and Africa are not in the Southern Hemisphere as some claim, but are crossed by the equator.</p><p>Arno Peters, a German filmmaker with a PhD in political propaganda, saw a great window of opportunity. Possibly drawing on his academic career, he held a series of lectures in which he presented Peters' projection, which he claimed was the most accurate representation of the world. And well, yes, this projection is equivalent, since it respects the areas of the different countries, but as a matter of compromise it completely destroys the shapes and distances.</p><p>The first obscure point of this story is that the projection he presented in Bonn in 1973 was not his own creation, but he was reusing a projection conceived by the cartographer James Gall in a scholarly paper in 1885. The second obscure point is that his work was by no means altruistic, for after convincing several countries and international institutions of the importance of adopting this new map, he began to charge for each of the reproductions sold. Today, in the midst of 2023, these reproductions still have a cost from which the heirs of Arno Peters continue to make a penny.</p><h3><strong>Winkel-Tripel, a good compromise</strong></h3><div class="captioned-image-container"><figure><a class="image-link image2 is-viewable-img" target="_blank" href="https://substackcdn.com/image/fetch/$s_!4sxo!,f_auto,q_auto:good,fl_progressive:steep/https%3A%2F%2Fsubstack-post-media.s3.amazonaws.com%2Fpublic%2Fimages%2Fc2c53808-9c87-45d9-b521-985e709a5d27_1024x679.jpeg" data-component-name="Image2ToDOM"><div class="image2-inset"><picture><source type="image/webp" srcset="https://substackcdn.com/image/fetch/$s_!4sxo!,w_424,c_limit,f_webp,q_auto:good,fl_progressive:steep/https%3A%2F%2Fsubstack-post-media.s3.amazonaws.com%2Fpublic%2Fimages%2Fc2c53808-9c87-45d9-b521-985e709a5d27_1024x679.jpeg 424w, https://substackcdn.com/image/fetch/$s_!4sxo!,w_848,c_limit,f_webp,q_auto:good,fl_progressive:steep/https%3A%2F%2Fsubstack-post-media.s3.amazonaws.com%2Fpublic%2Fimages%2Fc2c53808-9c87-45d9-b521-985e709a5d27_1024x679.jpeg 848w, https://substackcdn.com/image/fetch/$s_!4sxo!,w_1272,c_limit,f_webp,q_auto:good,fl_progressive:steep/https%3A%2F%2Fsubstack-post-media.s3.amazonaws.com%2Fpublic%2Fimages%2Fc2c53808-9c87-45d9-b521-985e709a5d27_1024x679.jpeg 1272w, https://substackcdn.com/image/fetch/$s_!4sxo!,w_1456,c_limit,f_webp,q_auto:good,fl_progressive:steep/https%3A%2F%2Fsubstack-post-media.s3.amazonaws.com%2Fpublic%2Fimages%2Fc2c53808-9c87-45d9-b521-985e709a5d27_1024x679.jpeg 1456w" sizes="100vw"><img src="https://substackcdn.com/image/fetch/$s_!4sxo!,w_1456,c_limit,f_auto,q_auto:good,fl_progressive:steep/https%3A%2F%2Fsubstack-post-media.s3.amazonaws.com%2Fpublic%2Fimages%2Fc2c53808-9c87-45d9-b521-985e709a5d27_1024x679.jpeg" width="725" height="480.7373046875" data-attrs="{&quot;src&quot;:&quot;https://substack-post-media.s3.amazonaws.com/public/images/c2c53808-9c87-45d9-b521-985e709a5d27_1024x679.jpeg&quot;,&quot;srcNoWatermark&quot;:null,&quot;fullscreen&quot;:null,&quot;imageSize&quot;:null,&quot;height&quot;:679,&quot;width&quot;:1024,&quot;resizeWidth&quot;:725,&quot;bytes&quot;:null,&quot;alt&quot;:&quot;Mapamundi de la National Geographic (2007)&quot;,&quot;title&quot;:null,&quot;type&quot;:null,&quot;href&quot;:null,&quot;belowTheFold&quot;:true,&quot;topImage&quot;:false,&quot;internalRedirect&quot;:null,&quot;isProcessing&quot;:false,&quot;align&quot;:null,&quot;offset&quot;:false}" class="sizing-normal" alt="Mapamundi de la National Geographic (2007)" title="Mapamundi de la National Geographic (2007)" srcset="https://substackcdn.com/image/fetch/$s_!4sxo!,w_424,c_limit,f_auto,q_auto:good,fl_progressive:steep/https%3A%2F%2Fsubstack-post-media.s3.amazonaws.com%2Fpublic%2Fimages%2Fc2c53808-9c87-45d9-b521-985e709a5d27_1024x679.jpeg 424w, https://substackcdn.com/image/fetch/$s_!4sxo!,w_848,c_limit,f_auto,q_auto:good,fl_progressive:steep/https%3A%2F%2Fsubstack-post-media.s3.amazonaws.com%2Fpublic%2Fimages%2Fc2c53808-9c87-45d9-b521-985e709a5d27_1024x679.jpeg 848w, https://substackcdn.com/image/fetch/$s_!4sxo!,w_1272,c_limit,f_auto,q_auto:good,fl_progressive:steep/https%3A%2F%2Fsubstack-post-media.s3.amazonaws.com%2Fpublic%2Fimages%2Fc2c53808-9c87-45d9-b521-985e709a5d27_1024x679.jpeg 1272w, https://substackcdn.com/image/fetch/$s_!4sxo!,w_1456,c_limit,f_auto,q_auto:good,fl_progressive:steep/https%3A%2F%2Fsubstack-post-media.s3.amazonaws.com%2Fpublic%2Fimages%2Fc2c53808-9c87-45d9-b521-985e709a5d27_1024x679.jpeg 1456w" sizes="100vw" loading="lazy"></picture><div class="image-link-expand"><div class="pencraft pc-display-flex pc-gap-8 pc-reset"><div class="pencraft pc-reset icon-container restack-image"><svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="lucide lucide-refresh-cw"><path d="M3 12a9 9 0 0 1 9-9 9.75 9.75 0 0 1 6.74 2.74L21 8"></path><path d="M21 3v5h-5"></path><path d="M21 12a9 9 0 0 1-9 9 9.75 9.75 0 0 1-6.74-2.74L3 16"></path><path d="M8 16H3v5"></path></svg></div><div class="pencraft pc-reset icon-container view-image"><svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="lucide lucide-maximize2"><polyline points="15 3 21 3 21 9"></polyline><polyline points="9 21 3 21 3 15"></polyline><line x1="21" x2="14" y1="3" y2="10"></line><line x1="3" x2="10" y1="21" y2="14"></line></svg></div></div></div></div></a><figcaption class="image-caption">(<a href="https://mapasmilhaud.com/mapas-politicos/mapamundi-de-la-national-geographic-2007/">source</a>)</figcaption></figure></div><p>Like almost everything in life, this is not a question of extremes. The two projections we have just seen focused on preserving only one of the three fundamental characteristics of maps, but it is possible to find a compromise that allows us to create generic cartographic representations without a specific goal.</p><p>This is one of the three projections proposed by the German cartographer Oswald Winkel in 1921. The name <em>tripel</em> comes from the German for <em>triple</em>, which refers to the fundamental objective of this projection, which is precisely to minimise the three fundamental distortions: area, angles and distance. Several studies by mathematicians and cartographers in the following decades raised this map among the projections that achieve the greatest compromise.</p><p>In 1998, thanks to this growing popularity, the Winkel-Tripel projection replaced the Robinson projection as the projection used for maps created by the National Geographic Society. In the past 25 years, more and more administrations, universities and educational institutions are following the example set by National Geographic and including this projection in their textbooks.</p><h3><strong>Dymaxion, a projection with a different objective</strong></h3><div class="captioned-image-container"><figure><a class="image-link image2 is-viewable-img" target="_blank" href="https://substackcdn.com/image/fetch/$s_!mfoh!,f_auto,q_auto:good,fl_progressive:steep/https%3A%2F%2Fsubstack-post-media.s3.amazonaws.com%2Fpublic%2Fimages%2F3c3adfa2-f82f-48e3-aa2f-b0f6637e3d6c_1692x1048.png" data-component-name="Image2ToDOM"><div class="image2-inset"><picture><source type="image/webp" srcset="https://substackcdn.com/image/fetch/$s_!mfoh!,w_424,c_limit,f_webp,q_auto:good,fl_progressive:steep/https%3A%2F%2Fsubstack-post-media.s3.amazonaws.com%2Fpublic%2Fimages%2F3c3adfa2-f82f-48e3-aa2f-b0f6637e3d6c_1692x1048.png 424w, https://substackcdn.com/image/fetch/$s_!mfoh!,w_848,c_limit,f_webp,q_auto:good,fl_progressive:steep/https%3A%2F%2Fsubstack-post-media.s3.amazonaws.com%2Fpublic%2Fimages%2F3c3adfa2-f82f-48e3-aa2f-b0f6637e3d6c_1692x1048.png 848w, https://substackcdn.com/image/fetch/$s_!mfoh!,w_1272,c_limit,f_webp,q_auto:good,fl_progressive:steep/https%3A%2F%2Fsubstack-post-media.s3.amazonaws.com%2Fpublic%2Fimages%2F3c3adfa2-f82f-48e3-aa2f-b0f6637e3d6c_1692x1048.png 1272w, https://substackcdn.com/image/fetch/$s_!mfoh!,w_1456,c_limit,f_webp,q_auto:good,fl_progressive:steep/https%3A%2F%2Fsubstack-post-media.s3.amazonaws.com%2Fpublic%2Fimages%2F3c3adfa2-f82f-48e3-aa2f-b0f6637e3d6c_1692x1048.png 1456w" sizes="100vw"><img src="https://substackcdn.com/image/fetch/$s_!mfoh!,w_1456,c_limit,f_auto,q_auto:good,fl_progressive:steep/https%3A%2F%2Fsubstack-post-media.s3.amazonaws.com%2Fpublic%2Fimages%2F3c3adfa2-f82f-48e3-aa2f-b0f6637e3d6c_1692x1048.png" width="1456" height="902" data-attrs="{&quot;src&quot;:&quot;https://substack-post-media.s3.amazonaws.com/public/images/3c3adfa2-f82f-48e3-aa2f-b0f6637e3d6c_1692x1048.png&quot;,&quot;srcNoWatermark&quot;:null,&quot;fullscreen&quot;:null,&quot;imageSize&quot;:null,&quot;height&quot;:902,&quot;width&quot;:1456,&quot;resizeWidth&quot;:null,&quot;bytes&quot;:3067056,&quot;alt&quot;:null,&quot;title&quot;:null,&quot;type&quot;:&quot;image/png&quot;,&quot;href&quot;:null,&quot;belowTheFold&quot;:true,&quot;topImage&quot;:false,&quot;internalRedirect&quot;:&quot;https://www.cartographerstale.com/i/167573976?img=https%3A%2F%2Fsubstack-post-media.s3.amazonaws.com%2Fpublic%2Fimages%2F3c3adfa2-f82f-48e3-aa2f-b0f6637e3d6c_1692x1048.png&quot;,&quot;isProcessing&quot;:false,&quot;align&quot;:null,&quot;offset&quot;:false}" class="sizing-normal" alt="" srcset="https://substackcdn.com/image/fetch/$s_!mfoh!,w_424,c_limit,f_auto,q_auto:good,fl_progressive:steep/https%3A%2F%2Fsubstack-post-media.s3.amazonaws.com%2Fpublic%2Fimages%2F3c3adfa2-f82f-48e3-aa2f-b0f6637e3d6c_1692x1048.png 424w, https://substackcdn.com/image/fetch/$s_!mfoh!,w_848,c_limit,f_auto,q_auto:good,fl_progressive:steep/https%3A%2F%2Fsubstack-post-media.s3.amazonaws.com%2Fpublic%2Fimages%2F3c3adfa2-f82f-48e3-aa2f-b0f6637e3d6c_1692x1048.png 848w, https://substackcdn.com/image/fetch/$s_!mfoh!,w_1272,c_limit,f_auto,q_auto:good,fl_progressive:steep/https%3A%2F%2Fsubstack-post-media.s3.amazonaws.com%2Fpublic%2Fimages%2F3c3adfa2-f82f-48e3-aa2f-b0f6637e3d6c_1692x1048.png 1272w, https://substackcdn.com/image/fetch/$s_!mfoh!,w_1456,c_limit,f_auto,q_auto:good,fl_progressive:steep/https%3A%2F%2Fsubstack-post-media.s3.amazonaws.com%2Fpublic%2Fimages%2F3c3adfa2-f82f-48e3-aa2f-b0f6637e3d6c_1692x1048.png 1456w" sizes="100vw" loading="lazy"></picture><div class="image-link-expand"><div class="pencraft pc-display-flex pc-gap-8 pc-reset"><div class="pencraft pc-reset icon-container restack-image"><svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="lucide lucide-refresh-cw"><path d="M3 12a9 9 0 0 1 9-9 9.75 9.75 0 0 1 6.74 2.74L21 8"></path><path d="M21 3v5h-5"></path><path d="M21 12a9 9 0 0 1-9 9 9.75 9.75 0 0 1-6.74-2.74L3 16"></path><path d="M8 16H3v5"></path></svg></div><div class="pencraft pc-reset icon-container view-image"><svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="lucide lucide-maximize2"><polyline points="15 3 21 3 21 9"></polyline><polyline points="9 21 3 21 3 15"></polyline><line x1="21" x2="14" y1="3" y2="10"></line><line x1="3" x2="10" y1="21" y2="14"></line></svg></div></div></div></div></a><figcaption class="image-caption">(<a href="https://www.geographicus.com/P/AntiqueMap/dymaxion-fullersadao-1954">source</a>)</figcaption></figure></div><p>The problem with the Winkel-Tripel projection is that it seeks extreme compromise, so it lacks an objective beyond the intention of becoming the most faithful representation possible. This may be necessary, but it leaves a wide range of areas where maps are needed, and other projections can be used.</p><p>Perhaps one of the most unique projections is the one created by Buckminster Fuller in 1946, which resulted in the Dymaxion map you can see above. To transfer the information from the sphere to the plane, this projection approximates the earth to an icosahedron and makes an independent projection on each of the 20 faces of the icosahedron. Once this initial step has been completed, all that remains is to develop the polyhedron on a plane in a way that is most useful for the use of the map.</p><div class="captioned-image-container"><figure><a class="image-link image2 is-viewable-img" target="_blank" href="https://substackcdn.com/image/fetch/$s_!kqhc!,f_auto,q_auto:good,fl_progressive:steep/https%3A%2F%2Fsubstack-post-media.s3.amazonaws.com%2Fpublic%2Fimages%2F2925c713-2220-40b6-a987-1dae6309b326_800x571.jpeg" data-component-name="Image2ToDOM"><div class="image2-inset"><picture><source type="image/webp" srcset="https://substackcdn.com/image/fetch/$s_!kqhc!,w_424,c_limit,f_webp,q_auto:good,fl_progressive:steep/https%3A%2F%2Fsubstack-post-media.s3.amazonaws.com%2Fpublic%2Fimages%2F2925c713-2220-40b6-a987-1dae6309b326_800x571.jpeg 424w, https://substackcdn.com/image/fetch/$s_!kqhc!,w_848,c_limit,f_webp,q_auto:good,fl_progressive:steep/https%3A%2F%2Fsubstack-post-media.s3.amazonaws.com%2Fpublic%2Fimages%2F2925c713-2220-40b6-a987-1dae6309b326_800x571.jpeg 848w, https://substackcdn.com/image/fetch/$s_!kqhc!,w_1272,c_limit,f_webp,q_auto:good,fl_progressive:steep/https%3A%2F%2Fsubstack-post-media.s3.amazonaws.com%2Fpublic%2Fimages%2F2925c713-2220-40b6-a987-1dae6309b326_800x571.jpeg 1272w, https://substackcdn.com/image/fetch/$s_!kqhc!,w_1456,c_limit,f_webp,q_auto:good,fl_progressive:steep/https%3A%2F%2Fsubstack-post-media.s3.amazonaws.com%2Fpublic%2Fimages%2F2925c713-2220-40b6-a987-1dae6309b326_800x571.jpeg 1456w" sizes="100vw"><img src="https://substackcdn.com/image/fetch/$s_!kqhc!,w_1456,c_limit,f_auto,q_auto:good,fl_progressive:steep/https%3A%2F%2Fsubstack-post-media.s3.amazonaws.com%2Fpublic%2Fimages%2F2925c713-2220-40b6-a987-1dae6309b326_800x571.jpeg" width="800" height="571" data-attrs="{&quot;src&quot;:&quot;https://substack-post-media.s3.amazonaws.com/public/images/2925c713-2220-40b6-a987-1dae6309b326_800x571.jpeg&quot;,&quot;srcNoWatermark&quot;:null,&quot;fullscreen&quot;:null,&quot;imageSize&quot;:null,&quot;height&quot;:571,&quot;width&quot;:800,&quot;resizeWidth&quot;:null,&quot;bytes&quot;:null,&quot;alt&quot;:null,&quot;title&quot;:null,&quot;type&quot;:null,&quot;href&quot;:null,&quot;belowTheFold&quot;:true,&quot;topImage&quot;:false,&quot;internalRedirect&quot;:null,&quot;isProcessing&quot;:false,&quot;align&quot;:null,&quot;offset&quot;:false}" class="sizing-normal" alt="" srcset="https://substackcdn.com/image/fetch/$s_!kqhc!,w_424,c_limit,f_auto,q_auto:good,fl_progressive:steep/https%3A%2F%2Fsubstack-post-media.s3.amazonaws.com%2Fpublic%2Fimages%2F2925c713-2220-40b6-a987-1dae6309b326_800x571.jpeg 424w, https://substackcdn.com/image/fetch/$s_!kqhc!,w_848,c_limit,f_auto,q_auto:good,fl_progressive:steep/https%3A%2F%2Fsubstack-post-media.s3.amazonaws.com%2Fpublic%2Fimages%2F2925c713-2220-40b6-a987-1dae6309b326_800x571.jpeg 848w, https://substackcdn.com/image/fetch/$s_!kqhc!,w_1272,c_limit,f_auto,q_auto:good,fl_progressive:steep/https%3A%2F%2Fsubstack-post-media.s3.amazonaws.com%2Fpublic%2Fimages%2F2925c713-2220-40b6-a987-1dae6309b326_800x571.jpeg 1272w, https://substackcdn.com/image/fetch/$s_!kqhc!,w_1456,c_limit,f_auto,q_auto:good,fl_progressive:steep/https%3A%2F%2Fsubstack-post-media.s3.amazonaws.com%2Fpublic%2Fimages%2F2925c713-2220-40b6-a987-1dae6309b326_800x571.jpeg 1456w" sizes="100vw" loading="lazy"></picture><div class="image-link-expand"><div class="pencraft pc-display-flex pc-gap-8 pc-reset"><div class="pencraft pc-reset icon-container restack-image"><svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="lucide lucide-refresh-cw"><path d="M3 12a9 9 0 0 1 9-9 9.75 9.75 0 0 1 6.74 2.74L21 8"></path><path d="M21 3v5h-5"></path><path d="M21 12a9 9 0 0 1-9 9 9.75 9.75 0 0 1-6.74-2.74L3 16"></path><path d="M8 16H3v5"></path></svg></div><div class="pencraft pc-reset icon-container view-image"><svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="lucide lucide-maximize2"><polyline points="15 3 21 3 21 9"></polyline><polyline points="9 21 3 21 3 15"></polyline><line x1="21" x2="14" y1="3" y2="10"></line><line x1="3" x2="10" y1="21" y2="14"></line></svg></div></div></div></div></a><figcaption class="image-caption">(<a href="https://commons.wikimedia.org/wiki/File:World_map_of_prehistoric_human_migrations.jpg">source</a>)</figcaption></figure></div><p>In this example of use, the icosahedron is developed with the aim of keeping all continents contiguous. Thanks to the peculiarity of this projection, we can easily see that all the continents of the Earth are located quite close to each other. This is probably the best projection to understand the migrations from Africa to the rest of the world. The arrows help with the direction the migrations followed, and the colours show the thousands of years that have passed since humans first arrived at each of the ends of the planet.</p><div><hr></div><p>I cannot conclude an article about projections without mentioning one of <a href="https://xkcd.com/977/">the best works done by xkcd</a>, which is this detailed, funny and quite accurate explanation of map projections. </p><div class="captioned-image-container"><figure><a class="image-link image2 is-viewable-img" target="_blank" href="https://substackcdn.com/image/fetch/$s_!MbGI!,f_auto,q_auto:good,fl_progressive:steep/https%3A%2F%2Fsubstack-post-media.s3.amazonaws.com%2Fpublic%2Fimages%2F2b423989-6212-47cf-a2a4-549ff7d0c99b_650x1990.png" data-component-name="Image2ToDOM"><div class="image2-inset"><picture><source type="image/webp" srcset="https://substackcdn.com/image/fetch/$s_!MbGI!,w_424,c_limit,f_webp,q_auto:good,fl_progressive:steep/https%3A%2F%2Fsubstack-post-media.s3.amazonaws.com%2Fpublic%2Fimages%2F2b423989-6212-47cf-a2a4-549ff7d0c99b_650x1990.png 424w, https://substackcdn.com/image/fetch/$s_!MbGI!,w_848,c_limit,f_webp,q_auto:good,fl_progressive:steep/https%3A%2F%2Fsubstack-post-media.s3.amazonaws.com%2Fpublic%2Fimages%2F2b423989-6212-47cf-a2a4-549ff7d0c99b_650x1990.png 848w, https://substackcdn.com/image/fetch/$s_!MbGI!,w_1272,c_limit,f_webp,q_auto:good,fl_progressive:steep/https%3A%2F%2Fsubstack-post-media.s3.amazonaws.com%2Fpublic%2Fimages%2F2b423989-6212-47cf-a2a4-549ff7d0c99b_650x1990.png 1272w, https://substackcdn.com/image/fetch/$s_!MbGI!,w_1456,c_limit,f_webp,q_auto:good,fl_progressive:steep/https%3A%2F%2Fsubstack-post-media.s3.amazonaws.com%2Fpublic%2Fimages%2F2b423989-6212-47cf-a2a4-549ff7d0c99b_650x1990.png 1456w" sizes="100vw"><img src="https://substackcdn.com/image/fetch/$s_!MbGI!,w_1456,c_limit,f_auto,q_auto:good,fl_progressive:steep/https%3A%2F%2Fsubstack-post-media.s3.amazonaws.com%2Fpublic%2Fimages%2F2b423989-6212-47cf-a2a4-549ff7d0c99b_650x1990.png" width="728" height="2228.8" data-attrs="{&quot;src&quot;:&quot;https://substack-post-media.s3.amazonaws.com/public/images/2b423989-6212-47cf-a2a4-549ff7d0c99b_650x1990.png&quot;,&quot;srcNoWatermark&quot;:null,&quot;fullscreen&quot;:null,&quot;imageSize&quot;:null,&quot;height&quot;:1990,&quot;width&quot;:650,&quot;resizeWidth&quot;:728,&quot;bytes&quot;:null,&quot;alt&quot;:&quot;Map Projections&quot;,&quot;title&quot;:null,&quot;type&quot;:null,&quot;href&quot;:null,&quot;belowTheFold&quot;:true,&quot;topImage&quot;:false,&quot;internalRedirect&quot;:null,&quot;isProcessing&quot;:false,&quot;align&quot;:null,&quot;offset&quot;:false}" class="sizing-normal" alt="Map Projections" title="Map Projections" srcset="https://substackcdn.com/image/fetch/$s_!MbGI!,w_424,c_limit,f_auto,q_auto:good,fl_progressive:steep/https%3A%2F%2Fsubstack-post-media.s3.amazonaws.com%2Fpublic%2Fimages%2F2b423989-6212-47cf-a2a4-549ff7d0c99b_650x1990.png 424w, https://substackcdn.com/image/fetch/$s_!MbGI!,w_848,c_limit,f_auto,q_auto:good,fl_progressive:steep/https%3A%2F%2Fsubstack-post-media.s3.amazonaws.com%2Fpublic%2Fimages%2F2b423989-6212-47cf-a2a4-549ff7d0c99b_650x1990.png 848w, https://substackcdn.com/image/fetch/$s_!MbGI!,w_1272,c_limit,f_auto,q_auto:good,fl_progressive:steep/https%3A%2F%2Fsubstack-post-media.s3.amazonaws.com%2Fpublic%2Fimages%2F2b423989-6212-47cf-a2a4-549ff7d0c99b_650x1990.png 1272w, https://substackcdn.com/image/fetch/$s_!MbGI!,w_1456,c_limit,f_auto,q_auto:good,fl_progressive:steep/https%3A%2F%2Fsubstack-post-media.s3.amazonaws.com%2Fpublic%2Fimages%2F2b423989-6212-47cf-a2a4-549ff7d0c99b_650x1990.png 1456w" sizes="100vw" loading="lazy"></picture><div class="image-link-expand"><div class="pencraft pc-display-flex pc-gap-8 pc-reset"><div class="pencraft pc-reset icon-container restack-image"><svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="lucide lucide-refresh-cw"><path d="M3 12a9 9 0 0 1 9-9 9.75 9.75 0 0 1 6.74 2.74L21 8"></path><path d="M21 3v5h-5"></path><path d="M21 12a9 9 0 0 1-9 9 9.75 9.75 0 0 1-6.74-2.74L3 16"></path><path d="M8 16H3v5"></path></svg></div><div class="pencraft pc-reset icon-container view-image"><svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="lucide lucide-maximize2"><polyline points="15 3 21 3 21 9"></polyline><polyline points="9 21 3 21 3 15"></polyline><line x1="21" x2="14" y1="3" y2="10"></line><line x1="3" x2="10" y1="21" y2="14"></line></svg></div></div></div></div></a><figcaption class="image-caption">(<a href="https://xkcd.com/977/">source</a>)</figcaption></figure></div><p class="button-wrapper" data-attrs="{&quot;url&quot;:&quot;https://www.cartographerstale.com/p/cartographic-projections?utm_source=substack&utm_medium=email&utm_content=share&action=share&quot;,&quot;text&quot;:&quot;Share&quot;,&quot;action&quot;:null,&quot;class&quot;:null}" data-component-name="ButtonCreateButton"><a class="button primary" href="https://www.cartographerstale.com/p/cartographic-projections?utm_source=substack&utm_medium=email&utm_content=share&action=share"><span>Share</span></a></p><div class="footnote" data-component-name="FootnoteToDOM"><a id="footnote-1" href="\#footnote-anchor-1" class="footnote-number" contenteditable="false" target="_self">1</a><div class="footnote-content"><p>I&#8217;ve already written about geodesics in this other article: <a href="https://www.cartographerstale.com/p/unexpected-curves-drawing-straight-a33">Unexpected curves: Drawing straight lines on a map</a>.</p></div></div>
        "#;

        let url = Url::parse("https://www.cartographerstale.com").unwrap();
        let res = clean_html_fragment(html, &url).unwrap();

        std::fs::write("/var/home/jangernert/test-output.html", &res.html).unwrap();
    }
}
